<?php

namespace Common\Files\Traits;

use Illuminate\Support\Str;

trait GetsEntryTypeFromMime
{
    protected function getTypeFromMime(
        string $mime,
        string $extension = null,
    ): string {
        $default = explode('/', $mime)[0];
        
        // Normalize extension to lowercase for comparison
        $extension = $extension ? strtolower($extension) : null;

        if ($mime === 'text/plain' && $extension === 'csv') {
            return 'spreadsheet';
        }

        // Handle HEIC/HEIF images
        if (in_array($extension, ['heic', 'heif']) || Str::contains($mime, 'heic') || Str::contains($mime, 'heif')) {
            return 'image';
        }

        // Handle MOV videos
        if ($extension === 'mov' || Str::contains($mime, 'quicktime')) {
            return 'video';
        }

        switch ($mime) {
            case 'application/x-zip-compressed':
            case 'application/zip':
                return 'archive';
            case 'application/pdf':
                return 'pdf';
            case 'image/svg':
                return 'image/svg+xml';
            case 'image/vnd.dwg':
            case 'image/tiff':
                // Fix: TIFF files should be classified as images, not generic files
                return 'image';
            case 'vnd.android.package-archive':
                return 'android package';
            case Str::contains($mime, ['xls', 'excel', 'spreadsheetml', 'csv']):
                return 'spreadsheet';
            case Str::contains($mime, 'photoshop'):
                return 'photoshop';
            case Str::contains($mime, 'officedocument.presentation'):
                return 'powerPoint';
            case Str::contains($mime, [
                'application/msword',
                'wordprocessingml.document',
            ]):
                return 'word';
            case Str::contains($mime, ['postscript', 'x-eps']):
                return 'postscript';
            case Str::startsWith($mime, 'message/rfc'):
                return 'text/plain';
            default:
                return $default === 'application' ? 'file' : $default;
        }
    }
}