<?php

namespace Common\Files\Services;

use Imagick;
use ImagickPixel;
use ImagickException;

class ConvertHeicToJpeg
{
    /**
     * Convert HEIC image to JPEG format
     *
     * @param resource|string $source Stream resource or file path
     * @return string JPEG binary data
     * @throws \Exception
     */
    public function convert($source): string
    {
        // If source is a stream resource, read it to a temporary file
        if (is_resource($source)) {
            $tempFile = tempnam(sys_get_temp_dir(), 'heic_');
            file_put_contents($tempFile, stream_get_contents($source));
            $result = $this->convertFile($tempFile);
            unlink($tempFile);
            return $result;
        } else {
            return $this->convertFile($source);
        }
    }
    
    /**
     * Convert HEIC and create thumbnail
     *
     * @param resource|string $source Stream resource or file path
     * @param int $width Thumbnail width
     * @param int $height Thumbnail height
     * @return string JPEG binary data
     * @throws \Exception
     */
    public function convertAndResize($source, int $width = 350, int $height = 250): string
    {
        // If source is a stream resource, read it to a temporary file
        if (is_resource($source)) {
            $tempFile = tempnam(sys_get_temp_dir(), 'heic_');
            file_put_contents($tempFile, stream_get_contents($source));
            $result = $this->convertFileAndResize($tempFile, $width, $height);
            unlink($tempFile);
            return $result;
        } else {
            return $this->convertFileAndResize($source, $width, $height);
        }
    }
    
    /**
     * Convert HEIC file to JPEG using multiple approaches
     *
     * @param string $source
     * @return string JPEG binary data
     * @throws \Exception
     */
    private function convertFile(string $source): string
    {
        // Try multiple approaches for converting HEIC files
        $exceptions = [];
        
        // Approach 1: Try with PHP Imagick extension
        try {
            return $this->convertWithImagick($source);
        } catch (\Exception $e) {
            $exceptions[] = "PHP Imagick: " . $e->getMessage();
        }
        
        // Approach 2: Try with ImageMagick command line tool
        try {
            return $this->convertWithCommandLine($source);
        } catch (\Exception $e) {
            $exceptions[] = "Command line: " . $e->getMessage();
        }
        
        // If all approaches fail, create a placeholder image
        return $this->createPlaceholderImage();
    }
    
    /**
     * Convert HEIC file and resize using multiple approaches
     *
     * @param string $source
     * @param int $width
     * @param int $height
     * @return string JPEG binary data
     * @throws \Exception
     */
    private function convertFileAndResize(string $source, int $width, int $height): string
    {
        // Try multiple approaches for converting HEIC files
        $exceptions = [];
        
        // Approach 1: Try with PHP Imagick extension
        try {
            return $this->convertAndResizeWithImagick($source, $width, $height);
        } catch (\Exception $e) {
            $exceptions[] = "PHP Imagick: " . $e->getMessage();
        }
        
        // Approach 2: Try with ImageMagick command line tool
        try {
            return $this->convertAndResizeWithCommandLine($source, $width, $height);
        } catch (\Exception $e) {
            $exceptions[] = "Command line: " . $e->getMessage();
        }
        
        // If all approaches fail, create a placeholder image
        return $this->createPlaceholderImage($width, $height);
    }
    
    /**
     * Convert HEIC to JPEG using PHP Imagick extension
     *
     * @param string $source
     * @return string JPEG binary data
     * @throws ImagickException
     */
    private function convertWithImagick(string $source): string
    {
        $imagick = new Imagick();
        
        try {
            // Try multiple approaches for reading HEIC files
            $this->readHeicImage($imagick, $source);
            
            // Auto-orient the image based on EXIF data
            $imagick->autoOrient();
            
            // Convert to JPEG
            $imagick->setImageFormat('jpeg');
            $imagick->setImageCompressionQuality(85);
            
            // Get the binary data
            $jpegData = $imagick->getImageBlob();
            
            return $jpegData;
        } finally {
            // Always clean up
            $imagick->clear();
            $imagick->destroy();
        }
    }
    
    /**
     * Convert HEIC to JPEG and resize using PHP Imagick extension
     *
     * @param string $source
     * @param int $width
     * @param int $height
     * @return string JPEG binary data
     * @throws ImagickException
     */
    private function convertAndResizeWithImagick(string $source, int $width, int $height): string
    {
        $imagick = new Imagick();
        
        try {
            // Try multiple approaches for reading HEIC files
            $this->readHeicImage($imagick, $source);
            
            // Try to auto-orient, but don't fail if it doesn't work
            try {
                $imagick->autoOrient();
            } catch (\Exception $e) {
                // Ignore orientation errors
            }
            
            // Resize to thumbnail
            $imagick->cropThumbnailImage($width, $height);
            
            // Convert to JPEG
            $imagick->setImageFormat('jpeg');
            $imagick->setImageCompressionQuality(85);
            
            // Get the binary data
            $jpegData = $imagick->getImageBlob();
            
            return $jpegData;
        } finally {
            // Always clean up
            $imagick->clear();
            $imagick->destroy();
        }
    }
    
    /**
     * Convert HEIC to JPEG using ImageMagick command line tool
     *
     * @param string $source
     * @return string JPEG binary data
     * @throws \Exception
     */
    private function convertWithCommandLine(string $source): string
    {
        $tempOutput = tempnam(sys_get_temp_dir(), 'jpeg_');
        
        try {
            // Try to convert using ImageMagick command line
            $command = "convert " . escapeshellarg($source) . " -quality 85 " . escapeshellarg($tempOutput);
            $output = [];
            $returnCode = 0;
            exec($command . " 2>&1", $output, $returnCode);
            
            if ($returnCode !== 0) {
                throw new \Exception("ImageMagick convert failed: " . implode("\n", $output));
            }
            
            if (!file_exists($tempOutput) || filesize($tempOutput) === 0) {
                throw new \Exception("ImageMagick convert produced no output");
            }
            
            $jpegData = file_get_contents($tempOutput);
            return $jpegData;
        } finally {
            if (file_exists($tempOutput)) {
                unlink($tempOutput);
            }
        }
    }
    
    /**
     * Convert HEIC to JPEG and resize using ImageMagick command line tool
     *
     * @param string $source
     * @param int $width
     * @param int $height
     * @return string JPEG binary data
     * @throws \Exception
     */
    private function convertAndResizeWithCommandLine(string $source, int $width, int $height): string
    {
        $tempOutput = tempnam(sys_get_temp_dir(), 'jpeg_');
        
        try {
            // Try to convert and resize using ImageMagick command line
            $command = "convert " . escapeshellarg($source) . " -resize " . escapeshellarg("${width}x${height}^") . " -gravity center -crop " . escapeshellarg("${width}x${height}+0+0") . " -quality 85 " . escapeshellarg($tempOutput);
            $output = [];
            $returnCode = 0;
            exec($command . " 2>&1", $output, $returnCode);
            
            if ($returnCode !== 0) {
                throw new \Exception("ImageMagick convert and resize failed: " . implode("\n", $output));
            }
            
            if (!file_exists($tempOutput) || filesize($tempOutput) === 0) {
                throw new \Exception("ImageMagick convert and resize produced no output");
            }
            
            $jpegData = file_get_contents($tempOutput);
            return $jpegData;
        } finally {
            if (file_exists($tempOutput)) {
                unlink($tempOutput);
            }
        }
    }
    
    /**
     * Try multiple approaches to read HEIC image
     *
     * @param Imagick $imagick
     * @param string $source
     * @throws ImagickException
     */
    private function readHeicImage(Imagick $imagick, string $source): void
    {
        $exceptions = [];
        
        // Approach 1: Try with no options
        try {
            $imagick->readImage($source);
            return;
        } catch (\Exception $e) {
            $exceptions[] = "No options: " . $e->getMessage();
        }
        
        // Approach 2: Try with preserve-orientation option
        try {
            $imagick->clear();
            $imagick->destroy();
            $imagick = new Imagick();
            $imagick->setOption('heic:preserve-orientation', 'true');
            $imagick->readImage($source);
            return;
        } catch (\Exception $e) {
            $exceptions[] = "Preserve orientation: " . $e->getMessage();
        }
        
        // Approach 3: Try with additional options
        try {
            $imagick->clear();
            $imagick->destroy();
            $imagick = new Imagick();
            $imagick->setOption('heic:preserve-orientation', 'true');
            $imagick->setOption('heic:preserve-colorprofile', 'true');
            $imagick->readImage($source);
            return;
        } catch (\Exception $e) {
            $exceptions[] = "Preserve orientation and color profile: " . $e->getMessage();
        }
        
        // Approach 4: Try with strict mode disabled
        try {
            $imagick->clear();
            $imagick->destroy();
            $imagick = new Imagick();
            $imagick->setOption('heic:strict', 'false');
            $imagick->readImage($source);
            return;
        } catch (\Exception $e) {
            $exceptions[] = "Strict mode disabled: " . $e->getMessage();
        }
        
        // Approach 5: Try with ping first
        try {
            $imagick->clear();
            $imagick->destroy();
            $imagick = new Imagick();
            $imagick->setOption('heic:strict', 'false');
            $imagick->pingImage($source); // Get info without loading pixel data
            $imagick->readImage($source);
            return;
        } catch (\Exception $e) {
            $exceptions[] = "Ping then read: " . $e->getMessage();
        }
        
        // Approach 6: Try forcing format
        try {
            $imagick->clear();
            $imagick->destroy();
            $imagick = new Imagick();
            $imagick->setOption('heic:strict', 'false');
            $imagick->setFormat('heic');
            $imagick->readImage($source);
            return;
        } catch (\Exception $e) {
            $exceptions[] = "Force format: " . $e->getMessage();
        }
        
        // If all approaches fail, throw the last exception with details
        $lastException = end($exceptions);
        throw new \Exception("Failed to read HEIC image with all approaches:\n" . implode("\n", $exceptions));
    }
    
    /**
     * Create a placeholder image when HEIC conversion fails
     *
     * @param int $width
     * @param int $height
     * @return string JPEG binary data
     */
    private function createPlaceholderImage(int $width = 350, int $height = 250): string
    {
        $placeholder = new Imagick();
        $placeholder->newImage($width, $height, new ImagickPixel('lightgray'));
        $placeholder->setImageFormat('jpeg');
        $placeholder->setImageCompressionQuality(85);
        $jpegData = $placeholder->getImageBlob();
        $placeholder->clear();
        $placeholder->destroy();
        return $jpegData;
    }
}