<?php

namespace Common\Files\Response;

use Common\Files\FileEntry;
use Common\Files\Response\FileResponse;
use Common\Files\Services\ConvertHeicToJpeg;
use Illuminate\Support\Str;
use Symfony\Component\HttpFoundation\StreamedResponse;

class HeicFileResponse implements FileResponse
{
    public function make(FileEntry $entry, array $options)
    {
        $downloadName = str_replace(
            ['%', '/'],
            '',
            pathinfo($entry->getNameWithExtension(), PATHINFO_FILENAME) . '.jpg'
        );

        $path = $entry->getStoragePath($options['useThumbnail']);
        $response = new StreamedResponse();
        $disposition = $response->headers->makeDisposition(
            $options['disposition'],
            $downloadName,
            Str::ascii($downloadName),
        );

        $response->headers->replace([
            'Content-Type' => 'image/jpeg',
            'Content-Disposition' => $disposition,
            'Cache-Control' => 'private, max-age=31536000, no-transform',
        ]);
        
        $response->setCallback(function () use ($entry, $path, $options) {
            $stream = $entry->getDisk()->readStream($path);
            if (!$stream) {
                abort(404);
            }

            $converter = new ConvertHeicToJpeg();
            
            try {
                if ($options['useThumbnail']) {
                    // Create a thumbnail (350x250) when thumbnail is requested
                    $jpegData = $converter->convertAndResize($stream, 350, 250);
                } else {
                    // Convert full image
                    $jpegData = $converter->convert($stream);
                }
                echo $jpegData;
            } catch (\Exception $e) {
                abort(500, 'Failed to convert HEIC image: ' . $e->getMessage());
            }
        });
        
        return $response;
    }
}