<?php namespace Common\Files\Response;

use Common\Files\FileEntry;
use Common\Files\Uploads\Uploads;
use League\Flysystem\AwsS3V3\AwsS3V3Adapter;
use League\Flysystem\Local\LocalFilesystemAdapter;

class FileResponseFactory
{
    public function create(
        FileEntry $entry,
        string $disposition = 'inline',
    ): mixed {
        // Check if this is a request for a HEIC thumbnail that doesn't exist
        $isThumbnailRequest = request('thumbnail');
        $isHeicFile = in_array(strtolower($entry->extension ?? ''), ['heic', 'heif']);
        
        $options = [
            'useThumbnail' => $isThumbnailRequest && $entry->thumbnail,
            'disposition' => $disposition,
        ];

        // If requesting a thumbnail for a HEIC file, use the HEIC response handler
        // even if thumbnail doesn't exist yet
        if ($isThumbnailRequest && $isHeicFile && $disposition === 'inline') {
            return $this->resolveResponseClass($entry, $disposition)->make(
                $entry,
                $options,
            );
        }

        return $this->resolveResponseClass($entry, $disposition)->make(
            $entry,
            $options,
        );
    }

    private function resolveResponseClass(
        FileEntry $entry,
        string $disposition = 'inline',
    ): FileResponse {
        // Handle HEIC/HEIF files with conversion FIRST, before checking remote URLs
        // This ensures HEIC files are always converted, even if stored on S3
        $extension = strtolower($entry->extension ?? '');
        if (in_array($extension, ['heic', 'heif']) && $disposition === 'inline') {
            return new HeicFileResponse();
        }

        $isLocalDrive =
            $entry->getDisk()->getAdapter() instanceof LocalFilesystemAdapter;
        $staticFileDelivery = config('filesystems.static_file_delivery');

        if ($this->shouldRedirectToRemoteUrl($entry)) {
            return new RemoteFileResponse();
        } elseif ($isLocalDrive && !$entry->public && $staticFileDelivery) {
            return $staticFileDelivery === 'xsendfile'
                ? new XSendFileResponse()
                : new XAccelRedirectFileResponse();
        } elseif (
            !$isLocalDrive &&
            config('filesystems.use_presigned_s3_urls')
        ) {
            return new StreamedFileResponse();
        } elseif (
            $disposition === 'inline' &&
            $this->shouldReturnRangeResponse($entry)
        ) {
            return new RangeFileResponse();
        } else {
            return new StreamedFileResponse();
        }
    }

    private function shouldReturnRangeResponse(FileEntry $entry): bool
    {
        return $entry->type === 'video' ||
            $entry->type === 'audio' ||
            $entry->mime === 'application/ogg';
    }

    private function shouldRedirectToRemoteUrl(FileEntry $entry): bool
    {
        $adapter = $entry->getDisk()->getAdapter();
        $isS3 = $adapter instanceof AwsS3V3Adapter;
        $shouldUsePublicUrl = $entry->public && $isS3;
        $shouldUsePresignedUrl =
            config('filesystems.use_presigned_s3_urls') && $isS3;
        $hasCustomCdnUrl =
            config('filesystems.file_preview_endpoint') ||
            ($entry->backend_id &&
                !!Uploads::backend($entry->backend_id)->customDomain);
        return $shouldUsePresignedUrl ||
            $shouldUsePublicUrl ||
            $hasCustomCdnUrl;
    }
}