<?php

namespace Common\Files\Listeners;

use Common\Files\Events\FileUploaded;
use Common\Files\FileEntry;
use Common\Files\Services\ConvertHeicToJpeg;
use Exception;
use Illuminate\Contracts\Queue\ShouldQueue;
use Intervention\Image\Drivers\Gd\Driver;
use Intervention\Image\ImageManager;

class CreateThumbnailForUploadedFile implements ShouldQueue
{
    public function handle(FileUploaded $event): void
    {
        // only create thumbnail for images over 500KB in size
        if (
            !$event->fileEntry->public &&
            $event->fileEntry->type === 'image' &&
            $event->fileEntry->file_size > 500000 &&
            !config('filesystems.disable_thumbnail_creation')
        ) {
            try {
                $this->maybeCreateThumbnail($event->fileEntry);
            } catch (Exception $e) {
                // Log the error but don't fail the job
                \Log::error('Thumbnail creation failed for file: ' . $event->fileEntry->id, [
                    'error' => $e->getMessage(),
                    'file_name' => $event->fileEntry->name,
                ]);
            }
        }
    }

    private function maybeCreateThumbnail(FileEntry $entry): void
    {
        $this->setMemoryLimit();
        $extension = strtolower($entry->extension ?? '');
        
        // Handle HEIC/HEIF files with Imagick conversion
        if (in_array($extension, ['heic', 'heif'])) {
            $this->createHeicThumbnail($entry);
            return;
        }
        
        $file = $entry->getDisk()->readStream($entry->getStoragePath());

        $manager = new ImageManager(new Driver());
        $img = $manager->read($file);

        $img->cover(350, 250);

        $encodedImg =
            $entry->extension === 'png' ? $img->toPng() : $img->toJpeg(60);
        $extension = $entry->extension === 'png' ? 'png' : 'jpg';

        $entry
            ->getDisk()
            ->put("{$entry->file_name}/thumbnail.$extension", $encodedImg, [
                'mimetype' => $encodedImg->mimetype(),
                'visibility' => 'public',
            ]);

        $entry->fill(['thumbnail' => true])->save();
    }
    
    private function createHeicThumbnail(FileEntry $entry): void
    {
        $file = $entry->getDisk()->readStream($entry->getStoragePath());
        $converter = new ConvertHeicToJpeg();
        
        try {
            $jpegData = $converter->convertAndResize($file, 350, 250);
            
            $entry
                ->getDisk()
                ->put("{$entry->file_name}/thumbnail.jpg", $jpegData, [
                    'mimetype' => 'image/jpeg',
                    'visibility' => 'public',
                ]);
            
            $entry->fill(['thumbnail' => true])->save();
        } catch (Exception $e) {
            // Even if thumbnail creation fails, log the error but don't throw
            // The file is still usable, and we've added fallback handling in the frontend
            \Log::error('HEIC thumbnail creation failed for file: ' . $entry->id, [
                'error' => $e->getMessage(),
                'file_name' => $entry->name,
            ]);
        }
    }

    private function setMemoryLimit(): void
    {
        $new = 512;
        $current = (int) ini_get('memory_limit');
        if ($current < $new) {
            @ini_set('memory_limit', "{$new}M");
        }
    }
}